package oracle.spatial.georaster.samples;

import com.sun.media.jai.codec.FileSeekableStream;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.ColorModel;
import java.awt.image.ComponentSampleModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.Raster;
import java.awt.image.SampleModel;
import java.awt.image.renderable.ParameterBlock;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.OutputStream;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import java.text.DecimalFormat;

import java.util.Properties;

import javax.media.jai.JAI;
import javax.media.jai.RenderedOp;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleResultSet;
import oracle.jdbc.OracleTypes;
import oracle.jdbc.pool.OracleDataSource;

import oracle.spatial.geometry.JGeometry;
import oracle.spatial.georaster.JGeoRaster;
import oracle.spatial.georaster.JGeoRasterMeta;
import oracle.spatial.georaster.RasterInfo;

import oracle.sql.BLOB;
import oracle.sql.CHAR;
import oracle.sql.CharacterSet;
import oracle.sql.NUMBER;
import oracle.sql.STRUCT;
import oracle.sql.StructDescriptor;

import oracle.xdb.XMLType;


/**
 * This is a sample java class, which shows how to implement a simple loader to load an image file
 * into a GeoRaster object based on GeoRaster Java API library. Note, this loader doesn't support 
 * user specified storage parameter, it will takes image's default setting for blocking,cellDepth
 * and so on. 
 * <p>
 */
public class Loader
{
    private JFrame m_frame;
    
    /**
     * constructor
     */    
    public Loader(JFrame frame)
    {
        m_frame=frame;
    }

    public void showDialog()
    {
        ImporterDialog dialog=new ImporterDialog(m_frame);
        dialog.show();
    }
    /**
     * This inner class is a customized Java JDialog to get input and load a specified image into 
     * database server.
     * <p>
     * @author  zzhang
     * @version 11
     */    
    class ImporterDialog extends JDialog implements ActionListener
    {
      JTextField m_hostName = new JTextField();
      JTextField m_DBName = new JTextField();
      JTextField m_port = new JTextField();
      JTextField m_userId = new JTextField();
      JTextField m_password = new JPasswordField();

      JTextField m_imageTable = new JTextField();
      JTextField m_columnName = new JTextField();
      JTextField m_rasterID = new JTextField();
      JTextField m_rdtName = new JTextField();

      JTextField m_rfileName = new JTextField();
      JButton m_rbrowse = new JButton();
      JTextField m_wfileName = new JTextField();
      JComboBox m_wfileType = new JComboBox();
      JButton m_wbrowse = new JButton();
      String m_fileName = null;

      String  m_wCurrentDir =  null;
      String  m_rCurrentDir =  null;

      public ImporterDialog(JFrame frame) throws HeadlessException
      {
        super();
        init(frame);
      }

      private void init(JFrame frame)
      {
        setModal(true);
        setTitle("Import to DB");
        setSize(500, 420);
        setResizable(false);

        JPanel tmpPanel = new JPanel(new BorderLayout());
        JPanel mainPanel = new JPanel(new GridBagLayout());
        mainPanel.setBorder(new javax.swing.border.EmptyBorder(6, 6, 0, 6));
        tmpPanel.add(mainPanel, BorderLayout.NORTH);

        JLabel hostNameL = new JLabel("Host name:");
        mainPanel.add(hostNameL, new GridBagConstraints(
            0, // gridx
            0, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_hostName.setText("localhost");
        mainPanel.add(m_hostName, new GridBagConstraints(
            1, // gridx
            0, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        hostNameL.setDisplayedMnemonic('h');
        hostNameL.setLabelFor(m_hostName);

        JLabel portL = new JLabel("Port number:");
        mainPanel.add(portL, new GridBagConstraints(
            0, // gridx
            1, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_port.setText("1521");
        mainPanel.add(m_port, new GridBagConstraints(
            1, // gridx
            1, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        portL.setDisplayedMnemonic('P');
        portL.setLabelFor(m_port);

        JLabel dbNameL = new JLabel("Database name:");
        mainPanel.add(dbNameL, new GridBagConstraints(
            0, // gridx
            2, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_DBName.setText("ORCL");
        mainPanel.add(m_DBName, new GridBagConstraints(
            1, // gridx
            2, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        dbNameL.setDisplayedMnemonic('D');
        dbNameL.setLabelFor(m_DBName);

        JLabel userIdL = new JLabel("User name: ");
        mainPanel.add(userIdL, new GridBagConstraints(
            0, // gridx
            3, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_userId, new GridBagConstraints(
            1, // gridx
            3, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        userIdL.setDisplayedMnemonic('U');
        userIdL.setLabelFor(m_userId);

        JLabel passwordL = new JLabel("Password: ");
        mainPanel.add(passwordL, new GridBagConstraints(
            0, // gridx
            4, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_password, new GridBagConstraints(
            1, // gridx
            4, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        passwordL.setDisplayedMnemonic('A');
        passwordL.setLabelFor(m_password);
        getContentPane().add(tmpPanel);

        JLabel imageTableL = new JLabel("Image table name:");
        mainPanel.add(imageTableL, new GridBagConstraints(
            0, // gridx
            5, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_imageTable, new GridBagConstraints(
            1, // gridx
            5, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        imageTableL.setDisplayedMnemonic('m');
        imageTableL.setLabelFor(m_imageTable);

        JLabel columnNameL = new JLabel("Column name:");
        mainPanel.add(columnNameL, new GridBagConstraints(
            0, // gridx
            6, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_columnName.setText("");
        mainPanel.add(m_columnName, new GridBagConstraints(
            1, // gridx
            6, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        columnNameL.setDisplayedMnemonic('c');
        columnNameL.setLabelFor(m_columnName);

        JLabel rasterIDL = new JLabel("Raster ID:");
        mainPanel.add(rasterIDL, new GridBagConstraints(
            0, // gridx
            7, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_rasterID.setText("");
        mainPanel.add(m_rasterID, new GridBagConstraints(
            1, // gridx
            7, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        rasterIDL.setDisplayedMnemonic('I');
        rasterIDL.setLabelFor(m_rasterID);

        JLabel rdtNameL = new JLabel("Rdt name:");
        mainPanel.add(rdtNameL, new GridBagConstraints(
            0, // gridx
            8, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        m_rdtName.setText("");
        mainPanel.add(m_rdtName, new GridBagConstraints(
            1, // gridx
            8, // gridy
            4, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        rdtNameL.setDisplayedMnemonic('n');
        rdtNameL.setLabelFor(m_rdtName);

        JLabel rfileNameL = new JLabel("Raster file name: ");
        mainPanel.add(rfileNameL, new GridBagConstraints(
            0, // gridx
            9, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_rfileName, new GridBagConstraints(
            1, // gridx
            9, // gridy
            2, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        rfileNameL.setDisplayedMnemonic('l');
        rfileNameL.setLabelFor(m_rfileName);
        mainPanel.add(m_rbrowse, new GridBagConstraints(
            3, // gridx
            9, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        m_rbrowse.setMnemonic('w');
        m_rbrowse.setText("Browse");
        m_rbrowse.addActionListener(this);

        m_wfileType.addItem("WORLDFILE");
        JLabel wfileTypeL = new JLabel("GeoRef file type: ");
        mainPanel.add(wfileTypeL, new GridBagConstraints(
            0, // gridx
            10, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_wfileType, new GridBagConstraints(
            1, // gridx
            10, // gridy
            3, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        wfileTypeL.setLabelFor(m_wfileType);


        JLabel wfileNameL = new JLabel("World file name: ");
        mainPanel.add(wfileNameL, new GridBagConstraints(
            0, // gridx
            11, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 6, 3, 3), // insets
            0, // ipadx
            0 // ipady
            ));
        mainPanel.add(m_wfileName, new GridBagConstraints(
            1, // gridx
            11, // gridy
            3, // gridwidth
            1, // gridheight
            1.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        wfileNameL.setDisplayedMnemonic('o');
        wfileNameL.setLabelFor(m_wfileName);
        mainPanel.add(m_wbrowse, new GridBagConstraints(
            4, // gridx
            11, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(3, 3, 3, 6), // insets
            0, // ipadx
            0 // ipady
            ));
        m_wbrowse.setMnemonic('r');
        m_wbrowse.setText("Browse");
        m_wbrowse.addActionListener(this);
        mainPanel.setBounds(6, 6, 6, 6);
        mainPanel.setBorder(BorderFactory.createRaisedBevelBorder());

        JPanel midPanel = new JPanel(new GridBagLayout());
        midPanel.setBorder(BorderFactory.createRaisedBevelBorder());

        JPanel butPanel = new JPanel(new GridBagLayout());
        JButton okButton = new JButton("  OK   ");
        JButton cancelButton = new JButton("Cancel");

        okButton.addActionListener(new java.awt.event.ActionListener() {
          public void actionPerformed(ActionEvent e)
          {
            okButton_actionPerformed(e);
          }
        });

        cancelButton.addActionListener(new java.awt.event.ActionListener() {
          public void actionPerformed(ActionEvent e)
          {
            dispose();
          }
        });

        butPanel.add(okButton, new GridBagConstraints(
            0, // gridx
            0, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(0, 0, 6, 3), // insets - 0,0,8,4
            0, // ipadx
            0 // ipady
            ));
        butPanel.add(cancelButton, new GridBagConstraints(
            1, // gridx
            0, // gridy
            1, // gridwidth
            1, // gridheight
            0.0D, // weightx
            0.0D, // weighty
            GridBagConstraints.CENTER, // anchor
            GridBagConstraints.HORIZONTAL, // fill
            new Insets(0, 3, 6, 6), // insets - 0,0,8,4
            0, // ipadx
            0 // ipady
            ));

        okButton.requestFocus();
        JPanel panel = new JPanel(new BorderLayout());
        panel.add(butPanel, BorderLayout.EAST);
        //tmpPanel.add(midPanel, BorderLayout.CENTER);
        panel.setBorder(BorderFactory.createRaisedBevelBorder());
        tmpPanel.add(panel, BorderLayout.SOUTH);
        getContentPane().add(tmpPanel);

        okButton.setDefaultCapable(true);
        rootPane.setDefaultButton(okButton);

        //initialize text field values
        try {
          Properties property = new Properties();
          property.load(new FileInputStream("GeoRasterToolImporter.properties"));
          if (property.getProperty("m_hostName") != null)
          {
            m_hostName.setText(property.getProperty("m_hostName"));
          }
          if (property.getProperty("m_DBName") != null)
          {
            m_DBName.setText(property.getProperty("m_DBName"));
          }
          if (property.getProperty("m_port") != null)
          {
            m_port.setText(property.getProperty("m_port"));
          }
          if (property.getProperty("m_userId") != null)
          {
            m_userId.setText(property.getProperty("m_userId"));
          }
          if (property.getProperty("m_imageTable") != null)
          {
            m_imageTable.setText(property.getProperty("m_imageTable"));
          }
          if (property.getProperty("m_columnName") != null)
          {
            m_columnName.setText(property.getProperty("m_columnName"));
          }
          if (property.getProperty("m_rasterID") != null)
          {
            m_rasterID.setText(property.getProperty("m_rasterID"));
          }
          if (property.getProperty("m_rdtName") != null)
          {
            m_rdtName.setText(property.getProperty("m_rdtName"));
          }
        }
        catch (Exception ee)
        {
        }
        if(m_fileName!=null)
        {
          this.m_rfileName.setText(m_fileName);
          this.m_rCurrentDir=m_fileName;
        }
        this.setLocation(frame.getLocation().x + 150, frame.getLocation().y + 150);
      }

    public void actionPerformed(ActionEvent e)
    {
        if(e.getSource()==m_wbrowse)
        {
            JFileChooser chooser=new JFileChooser(m_wCurrentDir);
            int returnVal=chooser.showOpenDialog(this);
            if(returnVal==JFileChooser.APPROVE_OPTION)
            {
                m_wfileName.setText(chooser.getSelectedFile().getAbsolutePath());
                m_wCurrentDir=
                        chooser.getCurrentDirectory().getAbsolutePath();
            }
        }
        else if(e.getSource()==m_rbrowse)
        {
            JFileChooser chooser=new JFileChooser(m_rCurrentDir);
            int returnVal=chooser.showOpenDialog(this);
            if(returnVal==JFileChooser.APPROVE_OPTION)
            {
                m_rfileName.setText(chooser.getSelectedFile().getAbsolutePath());
                m_rCurrentDir=chooser.getCurrentDirectory().getAbsolutePath();
            }
        }
    }

    private void okButton_actionPerformed(ActionEvent e)
    {
        JGeoRaster jgeor;
        // check the validity of the entered values.
        String msg=checkForNullValues();
        if(msg!=null)
        {
            JOptionPane.showConfirmDialog(this,"Missing values for : "+msg,
                                              "Error!",
                                              javax.swing.JOptionPane.DEFAULT_OPTION);
            return;
        }
        try
        {
            //store property values
            try
            {
                Properties property=new Properties();
                property.setProperty("m_hostName",m_hostName.getText());
                property.setProperty("m_port",m_port.getText());
                property.setProperty("m_DBName",m_DBName.getText());
                property.setProperty("m_userId",m_userId.getText());
                property.setProperty("m_imageTable",m_imageTable.getText());
                property.setProperty("m_columnName",m_columnName.getText());
                property.setProperty("m_rasterID",m_rasterID.getText());
                property.setProperty("m_rdtName",m_rdtName.getText());
                property.store(new FileOutputStream("GeoRasterToolImporter.properties"),"Loader");
            }
            catch(Exception ee)
            {
            }
            Connection conn=null;
            try
            {
                // Create a OracleDataSource instance
                OracleDataSource ods=new OracleDataSource();
                // Set the driver type
                ods.setDriverType("thin");
                // Set the database server name
                ods.setServerName(m_hostName.getText().trim());
                // Set the database name
                ods.setDatabaseName(m_DBName.getText().trim());
                // Set the port number
                ods.setPortNumber(new Integer((String)m_port.getText().trim()).intValue());
                // Set the user name
                ods.setUser(m_userId.getText().trim());
                // Set the password
                ods.setPassword(m_password.getText().trim());
                conn=ods.getConnection();
                conn.setAutoCommit(false);
            }
            catch(Exception eee)
            {
                //try service name
                String url=
                    "jdbc:oracle:thin:@(DESCRIPTION=(LOAD_BALANCE=on)(ADDRESS_LIST="+
                    "(ADDRESS=(PROTOCOL=TCP)(HOST="+
                    m_hostName.getText().trim()+") (PORT="+
                    m_port.getText().trim()+")))"+
                    "(CONNECT_DATA=(SERVICE_NAME="+
                    m_DBName.getText().trim()+")))";
                try
                {
                    OracleDataSource ods=new OracleDataSource();
                    ods.setDriverType("thin");
                    ods.setURL(url);
                    ods.setUser(m_userId.getText().trim());
                    ods.setPassword(m_password.getText().trim());
                    conn=ods.getConnection();
                    conn.setAutoCommit(false);
                }
                catch(Exception ee)
                {
                    ee.printStackTrace();
                    JOptionPane.showConfirmDialog(this,
                                                  "Can't make a connection to DB",
                                                  "Error!",
                                                  JOptionPane.DEFAULT_OPTION);
                    return;
                }
            }

            String rasterID=m_rasterID.getText().trim();
            String sqlString;
            OracleCallableStatement getData=null;
            sqlString="select a."+m_columnName.getText().trim()+" from "+m_imageTable.getText().trim()+
                        " a where a."+m_columnName.getText().trim()+"."+
                        "rasterID="+rasterID+" and a."+
                        m_columnName.getText().trim()+"."+"rasterDataTable='"+
                        m_rdtName.getText().toUpperCase().trim()+"'";
            PreparedStatement checkExisting=conn.prepareStatement(sqlString);
            ResultSet rset=checkExisting.executeQuery();
            OracleResultSet oraRset=(OracleResultSet)rset;
            if(oraRset.next())
            {
                oracle.sql.STRUCT oracleSTRUCT=(oracle.sql.STRUCT)oraRset.getObject(1);
                oracle.sql.Datum[] attrs=oracleSTRUCT.getOracleAttributes();
                if((attrs[0]!=null)||(attrs[1]!=null)||(attrs[4]!=null))
                {
                    Object[] options=new Object[]{ "OK","Cancel" };
                    int ret=JOptionPane.showOptionDialog(this,"An existing georaster object with specified RDT name and RID will be overwrited?",
                                                         "Warning",
                                                         JOptionPane.DEFAULT_OPTION,
                                                         JOptionPane.WARNING_MESSAGE,
                                                         null,options,
                                                         options[1]);
                    if(ret==JOptionPane.NO_OPTION)
                    {
                        return;
                    }
                }
                sqlString="declare\n"+"geor SDO_GEORASTER;\n"+"begin \n"+"delete from "+
                            m_rdtName.getText().trim()+" where rasterID="+
                            rasterID+";\n"+
                            "select a."+m_columnName.getText().trim()+" into geor from "+
                            m_imageTable.getText().trim()+" a where a."+
                            m_columnName.getText().trim()+"."+"rasterID="+
                            rasterID+" and a."+m_columnName.getText().trim()+
                            "."+"rasterDataTable='"+m_rdtName.getText().toUpperCase().trim()+
                            "' for update;\n"+
                            "geor:=sdo_geor.init('"+m_rdtName.getText().toUpperCase().trim()+"',"+
                            rasterID+");\n"
                            +"update "+m_imageTable.getText().trim()+" a set a."+
                            m_columnName.getText().trim()+"=geor where a."+
                            m_columnName.getText().trim()+".rasterID="+
                            rasterID+" and a."+m_columnName.getText().trim()+
                            ".rasterDataTable='"+m_rdtName.getText().toUpperCase().trim()+
                            "' returning a."+m_columnName.getText().trim()+
                            " into ? "+";\nend;";
                CallableStatement deleteRDT=conn.prepareCall(sqlString);
                deleteRDT.registerOutParameter(1,OracleTypes.STRUCT,"MDSYS.SDO_GEORASTER");
                rset=(OracleResultSet)deleteRDT.executeQuery();
                jgeor=new JGeoRaster((STRUCT)deleteRDT.getObject(1));   
            }
            else
            {
                JOptionPane.showConfirmDialog(this,
                                              "No existing GeoRaster Object with specified RDT name and RID",
                                              "Error!",
                                              javax.swing.JOptionPane.DEFAULT_OPTION);
                return;
            }

            //Load the input raster data image into a JAI RenderedOp object 
            //(or equivalent)
            FileSeekableStream inStream = new FileSeekableStream(m_rfileName.getText());
            /**
             * If source is a BLOB, you can create a SeekableStream as follows:
             * InputStream is = source.getBinaryStream();
             * SeekableStream inStream = SeekableStream.wrapInputStream(is, true);
             **/
            ParameterBlock params = new ParameterBlock();
            params.add(inStream);
            RenderedOp image = JAI.create("stream", params);

            //Parse the input raster data, raster block by raster block, and 
            //write the data to the SDO_RASTER objects corresponding to each 
            //raster block.
            int numXTiles = image.getNumXTiles();
            int numYTiles = image.getNumYTiles();
            int tileWidth = image.getTileWidth();
            int tileHeight = image.getTileHeight();
            int xTile, yTile, minX, minY;
            int bTile = 0; //Tiling along band dimension
            Raster tile = null;
            STRUCT blockObj = null, mbrObj = null;
            JGeometry blockMBR = null;
            int pyramidLevel = 0;
            
            //Stmt used to load each raster block individually into the target RDT
            //Returns the corresponding SDO_RASTER object.
            /*CallableStatement insertReturn = conn.prepareCall(
                                                        "begin\n "+
                                                        "INSERT INTO "+m_rdtName.getText().toUpperCase().trim() +
                                                        "\n VALUES (?, ?, ?, ?, ?, ?, EMPTY_BLOB()) "+
                                                        "\n RETURNING MDSYS.SDO_RASTER(rasterid, "+
                                                        " pyramidlevel,"+ 
                                                        " bandblocknumber,"+
                                                        " rowblocknumber, "+
                                                        " columnblocknumber," + 
                                                        " blockmbr,"+
                                                        " rasterblock) "+
                                                        "\n INTO ?;\n"+
                                                        "end;");*/
            
            for (xTile=0; xTile<numXTiles; xTile++)
            {
                for (yTile=0; yTile<numYTiles; yTile++)
                {
                    //insert the raster block into RDT with an empty blob
                    //retrieve the lob locator and populate it with raster data
                    tile = image.getTile(xTile, yTile); 
                    minX = tile.getMinX();
                    minY = tile.getMinY();
                    tileWidth = tile.getWidth();
                    tileHeight = tile.getHeight();
                    
                    blockMBR = new JGeometry(minY,minX,minY+tileHeight-1,minX+tileWidth-1,0);
                    BLOB blob=jgeor.getRasterObject().initRasterBlock(pyramidLevel,bTile,yTile,xTile,blockMBR);
                    writeRasterData(blob, tile);
                }
            }
            //insertReturn.close();
            initMetadata(image,jgeor);            
            if((m_wfileName.getText()!=null)&&(!m_wfileName.getText().trim().equalsIgnoreCase("")))
                loadWorldFile(m_wfileName.getText().trim(),jgeor);
            
            
            //update georaster object in the server side
	    jgeor.setRasterType(21001);
	    jgeor.storeToDB();
    

            conn.commit();
    
            JOptionPane.showConfirmDialog(this,
                                              "The GeoRaster object is loaded into DB successfully,\n Raster id is "+
                                              rasterID,"",
                                              JOptionPane.DEFAULT_OPTION);
            dispose();
        }
        catch(Exception ee)
       {
            JOptionPane.showConfirmDialog(this,ee.toString(),"Error!",
                                              javax.swing.JOptionPane.DEFAULT_OPTION);
            ee.printStackTrace();
            return;
        }
    }
    
    private void writeRasterData(BLOB rasterBlockBlob, Raster tile)throws Exception
    {
        OutputStream outStream = rasterBlockBlob.getBinaryOutputStream();
        int bufferSize = rasterBlockBlob.getBufferSize();
        byte[] outBuffer = new byte[bufferSize];
        int srcPos = 0; 
        int padSize= 0;
        
        int dataType, numBanks, bankSize;
        int[] offsets;
        int i = 0;
        DataBuffer rasterData = tile.getDataBuffer();       
        dataType = rasterData.getDataType();
        numBanks = rasterData.getNumBanks();
        bankSize = rasterData.getSize();
        offsets = rasterData.getOffsets();
        
        byte[] bank = new byte[bankSize];
        
        for (i=0; i < numBanks; i++)
        {
            srcPos = srcPos + offsets[i];
            bank = ((DataBufferByte)rasterData).getData(i);
            
            if(padSize > 0)
            {   
                System.arraycopy(bank,srcPos,outBuffer,bufferSize-padSize,padSize);
                outStream.write(outBuffer);
                srcPos = srcPos + padSize;
            } 
            
            while(srcPos <= bankSize-bufferSize)
            {
                System.arraycopy(bank,srcPos,outBuffer,0,bufferSize);
                outStream.write(outBuffer);
                srcPos = srcPos + bufferSize;
            }
            if(srcPos == bankSize)
                srcPos = 0; // start another bank
            else // what left must be less than one chunk
            {
                System.arraycopy(bank,srcPos,outBuffer,0,bankSize-srcPos);
                if( i == numBanks-1) //the last bank
                {
                    outStream.write(outBuffer,0,bankSize-srcPos);
                }
                else
                {      
                    //The size of data which needs to be padded 
                    //to the outBuffer from the next bank if 
                    //there is at least one more bank
                    padSize = bufferSize - (bankSize - srcPos);
                    srcPos = 0;
                }
            }
        }
        outStream.close();
    }
    
    
    // check any not-null constraint violations.
    // returns a string of comma delimited fields that are null

    private String checkForNullValues()
    {
        String data=new String();
        StringBuffer msg=new StringBuffer();
        int count=0;

        data=m_rfileName.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Raster file name");
            }
            else
            {
                msg.append("Raster file name");
            }
            count++;
        }

        data=m_hostName.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Host name");
            }
            else
            {
                msg.append("Host name");
            }
            count++;
        }

        data=m_port.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Port Number");
            }
            else
            {
                msg.append("Port Number");
            }
            count++;
        }

        data=m_DBName.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Database name");
            }
            else
            {
                msg.append("Database name");
            }
            count++;
        }

        data=m_userId.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"User ID");
            }
            else
            {
                msg.append("User ID");
            }
            count++;
        }
        data=m_password.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Password");
            }
            else
            {
                msg.append("Password");
            }
            count++;
        }

        data=m_imageTable.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Image table name");
            }
            else
            {
                msg.append("Image table name");
            }
            count++;
        }

        data=m_columnName.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"Column name");
            }
            else
            {
                msg.append("Column name");
            }
            count++;
        }

        data=m_rasterID.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"rasterID");
            }
            else
            {
                msg.append("rasterID");
            }
            count++;
        }

        data=m_rdtName.getText();
        if(data==null||data.trim().length()==0)
        {
            if(count!=0)
            {
                msg.append(", "+"rdt name");
            }
            else
            {
                msg.append("rdt name");
            }
            count++;
        }

        if(count==0)
        {
            return null;
        }
        else
        {
            return msg.toString();
        }
    }

    //generate metedata from an external image
    //assuming the whole image will be converted into GeoRaster object
    //in original format and size
    private  void initMetadata(RenderedOp image,JGeoRaster geor) throws Exception
    {
    
        JGeoRasterMeta meta = geor.getMetadataObject();
    
        if(image == null) 
            return;
    
        //set metadata
        SampleModel sampleModel = image.getSampleModel();
        ColorModel  colorModel  = image.getColorModel();
    
        

        long band = image.getNumBands();
        long column = image.getWidth();
        long row = image.getHeight();
        String rasterSpec="dimSize=("+row+","+column+","+band+") ";
        
        String sm_class = sampleModel.getClass().getName();
        String sampleModelType = sm_class;
        int dataType = sampleModel.getDataType();
        String rasterInfo_cellDepth_text="";
        switch(dataType)
        {
            case DataBuffer.TYPE_BYTE:
                rasterInfo_cellDepth_text = "8BIT_U";
            break;
            case DataBuffer.TYPE_USHORT:
                rasterInfo_cellDepth_text = "16BIT_U";
            break;
            case DataBuffer.TYPE_INT:
                rasterInfo_cellDepth_text = "32BIT_U";
            break;
            case DataBuffer.TYPE_SHORT:
                rasterInfo_cellDepth_text = "16BIT_S";
            break;
            case DataBuffer.TYPE_FLOAT:
                rasterInfo_cellDepth_text = "32BIT_REAL";
            break;
            case DataBuffer.TYPE_DOUBLE:
                rasterInfo_cellDepth_text = "64BIT_REAL";
            break;
        }        
        rasterSpec=rasterSpec+"cellDepth="+rasterInfo_cellDepth_text+" ";
        rasterSpec=rasterSpec+"ultCoord=("+image.getMinY()+","+image.getMinX()+",0) "; 
        rasterSpec=rasterSpec+"compression=NONE ";

        //meta.colorMap = GeoRasterMetadata.extractColorMap(colorModel);
   
        int numYTiles = image.getNumYTiles();
        int numXTiles = image.getNumXTiles();
        
        int rasterInfo_blocking_totalRowBlocks = numYTiles;
        int rasterInfo_blocking_totalColumnBlocks = numXTiles;
        int rasterInfo_blocking_totalBandBlocks = 1; 
        int rowBlockSize = image.getTileHeight();
        int columnBlockSize = image.getTileWidth();
        int bandBlockSize = image.getNumBands();        
        if(rasterInfo_blocking_totalRowBlocks == 1 &&rasterInfo_blocking_totalColumnBlocks == 1)
            rasterSpec=rasterSpec+"blocking=FALSE ";        
        else
            rasterSpec=rasterSpec+"blocking=TRUE blocksize=("+rowBlockSize+","+columnBlockSize+","+bandBlockSize+") ";        
        
        if(sampleModelType.startsWith("java.awt.image.PixelInterleavedSampleModel"))
            rasterSpec=rasterSpec+"interleaving=BIP ";
        else if(sampleModelType.startsWith("java.awt.image.BandedSampleModel"))
            rasterSpec=rasterSpec+"interleaving=BSQ ";
        rasterSpec=rasterSpec+"rLevel=0 ";
        meta.init(21001,rasterSpec);
        meta.getRasterInfo().setPyramidType(RasterInfo.PYRAMID_NONE);
        //check colorModel to get the default color info
        if(band >= 3)
        {
            if (sampleModel == null)
            {
                meta.getObjectInfo().setDefaultRed(new Long(1));
                meta.getObjectInfo().setDefaultGreen(new Long(2));
                meta.getObjectInfo().setDefaultBlue(new Long(3));
            }
            else
            {
                int[] offsets = ((ComponentSampleModel)sampleModel).getBandOffsets();
                meta.getObjectInfo().setDefaultRed(new Long(offsets[0]+1));
                meta.getObjectInfo().setDefaultGreen(new Long(offsets[1]+1));
                meta.getObjectInfo().setDefaultBlue(new Long(offsets[2]+1));                
            }
        }
    }
    public boolean loadWorldFile(String srs,JGeoRaster jgeor )throws Exception
    {
        DecimalFormat decf = (DecimalFormat)DecimalFormat.getInstance();
        FileReader srsSource=new FileReader(new File(srs));
        BufferedReader srsFile;
        try
        {
            double A, D, B, E, C, F;
            srsFile = new BufferedReader(srsSource);
            A = decf.parse(srsFile.readLine().trim()).doubleValue();
            D = decf.parse(srsFile.readLine().trim()).doubleValue();
            B = decf.parse(srsFile.readLine().trim()).doubleValue();
            E = decf.parse(srsFile.readLine().trim()).doubleValue();
            C = decf.parse(srsFile.readLine().trim()).doubleValue();
            F = decf.parse(srsFile.readLine().trim()).doubleValue();
            jgeor.getMetadataObject().getSpatialReferenceInfo().setWorldFile(A,B,C,D,E,F);
        }
        catch (Exception e)
        {
            throw e;
        }
        return true;
    }
    
}

}
